/**
 *  Copyright (c) 2012 BMW
 *
 *  \author Christian Mueller, christian.ei.mueller@bmw.de BMW 2011,2012
 *
 *  \copyright
 *  Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction,
 *  including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 *  subject to the following conditions:
 *  The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 *  THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *  For further information see http://www.genivi.org/.
 */

/*
 * DO NOT REMOVE ABOVE COPYRIGHT - Ranjit
 */

#include "g3g_dbus.h"
#include <cstdlib>
#include <cassert>
#include <vector>
#include <iostream>
#include <string>

/*
 * Constructor for use by a dbus adaptor object
 */
g3g_dbusmessagehandler::g3g_dbusmessagehandler(DBusConnection* connection, std::string ifname, std::string objpath ) :
        mpDBusConnection(connection),
        msInterfaceName(ifname),
        msPath(objpath),
        msNodeName()
{
}
/*
 * constructor for use by a dbus proxy object
 */
g3g_dbusmessagehandler::g3g_dbusmessagehandler(DBusConnection* connection, std::string nodename, std::string ifname, std::string objpath )
:mpDBusConnection(connection),
msInterfaceName(ifname),
msPath(objpath),
msNodeName(nodename)
{
}

g3g_dbusmessagehandler::~g3g_dbusmessagehandler(){
	mpDBusConnection = NULL;
}

g3g_dbusmessage* g3g_dbusmessagehandler::CreateReply(g3g_dbusmessage* msg) const
{
	if(msg!=NULL)
	{
		DBusMessage* mpDBusMessage = dbus_message_new_method_return(msg->getDbusMsgPtr());

		g3g_dbusmessage* reply = new g3g_dbusmessage(mpDBusMessage,true, dbus_message_get_serial(mpDBusMessage));

		return reply;
	}
	return NULL;
}

g3g_dbusmessage* g3g_dbusmessagehandler::CreateSignal(std::string signalName) const
{
	if(msPath.empty() || signalName.empty() || msInterfaceName.empty())
	{
		std::cout<<"One of Path, Signal Name, Interface name is empty"<<std::endl;
		return NULL;
	}

	DBusMessage* mpDBusMessage = dbus_message_new_signal(msPath.c_str(), msInterfaceName.c_str(), signalName.c_str());

	g3g_dbusmessage* dbussignal = new g3g_dbusmessage(mpDBusMessage,true);

    return dbussignal;
}
g3g_dbusmessage* g3g_dbusmessagehandler::CreateSignal_Basic(std::string interfacename, std::string objectpath,std::string signalName) const
{
	if(interfacename.empty() || signalName.empty() || objectpath.empty())
	{
		std::cout<<"One of Path, Signal Name, Interface name is empty"<<std::endl;
		return NULL;
	}

	DBusMessage* mpDBusMessage = dbus_message_new_signal(objectpath.c_str(), interfacename.c_str(), signalName.c_str());

	g3g_dbusmessage* dbussignal = new g3g_dbusmessage(mpDBusMessage,true);

    return dbussignal;
}

g3g_dbusmessage* g3g_dbusmessagehandler::CreateMethodCall(std::string methodname) const
{
	if(msPath.empty() || methodname.empty() || msInterfaceName.empty() || msNodeName.empty())
	{
		std::cout<<"One of Node Name, Interface name, Path or Method Name is empty"<<std::endl;
		return NULL;
	}
	DBusMessage* mpDBusMessage = dbus_message_new_method_call(msNodeName.c_str(), msPath.c_str(), msInterfaceName.c_str(), methodname.c_str());

	g3g_dbusmessage* method_call = new g3g_dbusmessage(mpDBusMessage,true);

	return method_call;
}

g3g_dbusmessage* g3g_dbusmessagehandler::CreateMethodCall_Basic(std::string busname, std::string objectpath, std::string interfacename,std::string MethodName) const
{
	if(busname.empty() || objectpath.empty() || interfacename.empty() || MethodName.empty())
	{
		std::cout<<"One of bus Name, Interface name, Path or Method Name is empty"<<std::endl;
		return NULL;
	}
	DBusMessage* mpDBusMessage = dbus_message_new_method_call(busname.c_str(), objectpath.c_str(), interfacename.c_str(), MethodName.c_str());

	g3g_dbusmessage* method_call = new g3g_dbusmessage(mpDBusMessage,true);

	return method_call;
}


void g3g_dbusmessagehandler::sendSignal(g3g_dbusmessage* sgnal)
{
    assert(mpDBusConnection!=NULL);
    if (sgnal != NULL)
    {
    	dbus_uint32_t serial = 1;

        if (!dbus_connection_send(mpDBusConnection, sgnal->getDbusMsgPtr(), &serial))
        {
        	std::cout<<"DBusMessageHandler::sendMessage cannot send message!"<<std::endl;
        }
        dbus_connection_flush(mpDBusConnection);
        dbus_message_unref(sgnal->mpDBusMessage);
        sgnal->mpDBusMessage = NULL;
    }
}

void g3g_dbusmessagehandler::sendMethodReply(g3g_dbusmessage* method_call, g3g_dbusmessage* method_reply)
{
	if((method_call == NULL) || (method_reply == NULL))
	{
		return;
	}

    assert(mpDBusConnection!=NULL);

	dbus_uint32_t serial = 1;

	DBusMessage* msg = method_reply->mpDBusMessage;

	serial = dbus_message_get_serial(method_call->mpDBusMessage);//Get the serial from method call

	//If you have any errors, send error message instead
	if(!method_reply->mErrorName.empty())
	{
		dbus_message_unref(msg);//Free memory for reply message

		msg = dbus_message_new_error(method_call->mpDBusMessage, method_reply->mErrorName.c_str(), method_reply->mErrorMsg.c_str());
	}
    if (!dbus_connection_send(mpDBusConnection, msg, &serial))
    {
    	std::cout<<"DBusMessageHandler::sendMessage cannot send message!"<<std::endl;
    }
    dbus_connection_flush(mpDBusConnection);
    dbus_message_unref(msg);
    msg = NULL;
}

g3g_dbusmessage* g3g_dbusmessagehandler::sendMethodCall(g3g_dbusmessage* method_call)
{
	assert(mpDBusConnection!=NULL);

	DBusPendingCall* pending;

    /*
     * Blocking call ... remove it in next versions
     */
    if (!dbus_connection_send_with_reply(mpDBusConnection, method_call->mpDBusMessage, &pending, -1))
    {
        std::cout<<"DBusMessageHandler::sendMessage cannot send message!"<<std::endl;
    }
    dbus_connection_flush(mpDBusConnection);
    dbus_message_unref(method_call->mpDBusMessage);
    method_call->mpDBusMessage = NULL;

    if(NULL != pending)
    {
        //Now block on the pending call
        dbus_pending_call_block(pending);

        //Get the reply message from the queue
        DBusMessage *reply = dbus_pending_call_steal_reply(pending);

        //Free pending call handle
        dbus_pending_call_unref(pending);

        if(NULL == reply)
        {
        	std::cout<< "Received Null as method reply"<<std::endl;
        }
        int reply_type = dbus_message_get_type(reply);
        switch(reply_type)
        {
        case DBUS_MESSAGE_TYPE_METHOD_RETURN:
        {
        	g3g_dbusmessage* response = new g3g_dbusmessage(reply);
        	return response;
        }
        break;//Not executed anyway
        case DBUS_MESSAGE_TYPE_ERROR:
        {
        	std::string error = dbus_message_get_error_name(reply);
        	std::cout<<"Error Response : "<<error<<std::endl;
        }
        break;
        default://DoesNot happen at all
        	break;
        }
    }
    else
    {
    	std::cout<<"Connection with Server disconnected"<<std::endl;
    }
    return NULL;
}

void g3g_dbusmessagehandler::sendMethodCallNoReply(g3g_dbusmessage* method_call)
{
	assert(mpDBusConnection!=NULL);

    /*
     * Non Blocking call
     */
    if (!dbus_connection_send(mpDBusConnection, method_call->mpDBusMessage, &method_call->mSerial))
    {
        std::cout<<"DBusMessageHandler::sendMessage cannot send message!"<<std::endl;
    }

    dbus_connection_flush(mpDBusConnection);
    dbus_message_unref(method_call->mpDBusMessage);
    method_call->mpDBusMessage = NULL;
}

/*
 * DBus Handler Message
 */
g3g_dbusmessage::g3g_dbusmessage(DBusMessage *ptr, bool bWriteMode, dbus_uint32_t reply_serial)
:mDBusError(),
 mSerial(reply_serial),
 mErrorName(),
 mErrorMsg(),
 mpDBusMessage(ptr),
 mMessageIterList()
{
	if(ptr != NULL)
	{
	    DBusMessageIter* temp = new DBusMessageIter;

	    if(bWriteMode)//Write mode
	    {
	    	dbus_message_iter_init_append(mpDBusMessage, temp);
	    }
	    else //Read only mode
	    {
	    	dbus_message_iter_init(ptr,temp);
	    }
	    mMessageIterList.push_back(temp);
		//temp = NULL;//Satisfying lint
	}
	else
	{
		std::cout<<"message pointer is null"<<std::endl;
	}
}

void g3g_dbusmessage::vClearMessageIterList()
{
	while(!mMessageIterList.empty())
	{
		DBusMessageIter* temp = mMessageIterList.back();
		if(temp != NULL)
		{
			delete temp;
		}
		mMessageIterList.pop_back();
	}
}

char* g3g_dbusmessage::getString()
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return NULL;
	}
    char* param = NULL;

    if (DBUS_TYPE_STRING != dbus_message_iter_get_arg_type(mMessageIterList.back()))
    {
        std::cout<<"DBusMessageHandler::getString DBUS handler argument is no String!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no string";
    }
    else
    {
        dbus_message_iter_get_basic(mMessageIterList.back(), &param);
        dbus_message_iter_next(mMessageIterList.back());
    }
    return (param);
}

dbus_bool_t g3g_dbusmessage::getBool()
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return false;
	}
    dbus_bool_t boolparam = false;

    if (DBUS_TYPE_BOOLEAN != dbus_message_iter_get_arg_type(mMessageIterList.back()))
    {
        std::cout<<"DBusMessageHandler::getBool DBUS handler argument is no bool!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no bool";
    }
    else
    {
        dbus_message_iter_get_basic(mMessageIterList.back(), &boolparam);
        dbus_message_iter_next(mMessageIterList.back());
    }
    return (boolparam);
}

char g3g_dbusmessage::getByte()
{
	char param = 0;

	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return param;
	}

    if (DBUS_TYPE_BYTE != dbus_message_iter_get_arg_type(mMessageIterList.back()))
    {
        std::cout<<"DBusMessageHandler::getByte DBUS handler argument is no byte!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no byte";
    }
    else
    {
        dbus_message_iter_get_basic(mMessageIterList.back(), &param);
        dbus_message_iter_next(mMessageIterList.back());
    }
    return (param);
}

dbus_uint16_t g3g_dbusmessage::getUInt16()
{
    dbus_uint16_t param=0;

	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return 0;
	}
#ifdef GLIB_DBUS_TYPES_TOLERANT
    if (DBUS_TYPE_UINT16 != dbus_message_iter_get_arg_type(mMessageIterList.back()) && DBUS_TYPE_UINT32 != dbus_message_iter_get_arg_type(mMessageIterList.back()))
#else
    if (DBUS_TYPE_UINT16 != dbus_message_iter_get_arg_type(mMessageIterList.back()))
#endif
    {
        std::cout<<"DBusMessageHandler::getUInt DBUS handler argument is no uint16_t!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no uint16_t";
    }
    else
    {
        dbus_message_iter_get_basic(mMessageIterList.back(), &param);
        dbus_message_iter_next(mMessageIterList.back());
    }
    return (param);
}

dbus_int16_t g3g_dbusmessage::getInt16()
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return 0;
	}
    dbus_int16_t param=0;

#ifdef GLIB_DBUS_TYPES_TOLERANT
    if (DBUS_TYPE_INT16 != dbus_message_iter_get_arg_type(mMessageIterList.back()) && DBUS_TYPE_INT32 != dbus_message_iter_get_arg_type(mMessageIterList.back()))
#else
    if (DBUS_TYPE_INT16 != dbus_message_iter_get_arg_type(mMessageIterList.back()))
#endif
    {
        std::cout<<"DBusMessageHandler::getInt DBUS handler argument is no int16_t!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no int16_t";
    }
    else
    {
        dbus_message_iter_get_basic(mMessageIterList.back(), &param);
        dbus_message_iter_next(mMessageIterList.back());
    }
    return (param);
}

dbus_uint32_t g3g_dbusmessage::getUInt32()
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return 0;
	}
    dbus_uint32_t param=0;

    if (DBUS_TYPE_UINT32 != dbus_message_iter_get_arg_type(mMessageIterList.back()))
    {
        std::cout<<"DBusMessageHandler::getUInt DBUS handler argument is no uint32_t!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no uint32_t";
    }
    else
    {
        dbus_message_iter_get_basic(mMessageIterList.back(), &param);
        dbus_message_iter_next(mMessageIterList.back());
    }
    return (param);
}

dbus_int32_t g3g_dbusmessage::getInt32()
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return 0;
	}
    dbus_int32_t param=0;

    if (DBUS_TYPE_INT32 != dbus_message_iter_get_arg_type(mMessageIterList.back()))
    {
        std::cout<<"DBusMessageHandler::getInt DBUS handler argument is no int32_t!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no int32_t";
    }
    else
    {
        dbus_message_iter_get_basic(mMessageIterList.back(), &param);
        dbus_message_iter_next(mMessageIterList.back());
    }
    return (param);
}

dbus_uint64_t g3g_dbusmessage::getUInt64()
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return 0;
	}
    dbus_uint64_t param=0;

    if (DBUS_TYPE_UINT64 != dbus_message_iter_get_arg_type(mMessageIterList.back()))
    {
        std::cout<<"DBusMessageHandler::getUInt DBUS handler argument is no uint64_t!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no uint64_t";
    }
    else
    {
        dbus_message_iter_get_basic(mMessageIterList.back(), &param);
        dbus_message_iter_next(mMessageIterList.back());
    }
    return (param);
}

dbus_int64_t g3g_dbusmessage::getInt64()
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return 0;
	}
    dbus_int64_t param=0;

    if (DBUS_TYPE_INT64 != dbus_message_iter_get_arg_type(mMessageIterList.back()))
    {
        std::cout<<"DBusMessageHandler::getInt DBUS handler argument is no int64_t!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no int64_t";
    }
    else
    {
        dbus_message_iter_get_basic(mMessageIterList.back(), &param);
        dbus_message_iter_next(mMessageIterList.back());
    }
    return (param);
}

double g3g_dbusmessage::getDouble()
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return 0;
	}
    double param=0;
    if (DBUS_TYPE_DOUBLE != dbus_message_iter_get_arg_type(mMessageIterList.back()))
    {
        std::cout<<"DBusMessageHandler::getDouble DBUS handler argument is no double!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no double";
    }
    else
    {
        dbus_message_iter_get_basic(mMessageIterList.back(), &param);
        dbus_message_iter_next(mMessageIterList.back());
    }
    return (param);
}

void g3g_dbusmessage::getProperty(dbus_int16_t & type, dbus_int16_t & value)
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return;
	}
    DBusMessageIter arrayIter;
    if (DBUS_TYPE_STRUCT != dbus_message_iter_get_arg_type(mMessageIterList.back()))
    {
        std::cout<<"DBusMessageHandler::getProperty DBUS handler argument is no array!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no array";
    }
    else
    {
        dbus_message_iter_recurse(mMessageIterList.back(), &arrayIter);
        dbus_message_iter_get_basic(&arrayIter, &type);
        dbus_message_iter_next(&arrayIter);
        dbus_message_iter_get_basic(&arrayIter, &value);
        dbus_message_iter_next(mMessageIterList.back());
    }
}

void g3g_dbusmessage::append(bool toAppend)
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return;
	}
    dbus_bool_t mybool = toAppend;
    if (!dbus_message_iter_append_basic(mMessageIterList.back(), DBUS_TYPE_BOOLEAN, &mybool))
    {
        std::cout<<"DBusMessageHandler::append Cannot allocate DBus message!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void g3g_dbusmessage::append(double toAppend)
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return;
	}
    if (!dbus_message_iter_append_basic(mMessageIterList.back(), DBUS_TYPE_DOUBLE, &toAppend))
    {
        std::cout<<"DBusMessageHandler::append Cannot allocate DBus message!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void g3g_dbusmessage::append(char toAppend)
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return;
	}
    if (!dbus_message_iter_append_basic(mMessageIterList.back(), DBUS_TYPE_BYTE, &toAppend))
    {
        std::cout<<"DBusMessageHandler::append Cannot allocate DBus message!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}
void g3g_dbusmessage::append(uint8_t toAppend)
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return;
	}
    if (!dbus_message_iter_append_basic(mMessageIterList.back(), DBUS_TYPE_BYTE, &toAppend))
    {
        std::cout<<"DBusMessageHandler::append Cannot allocate DBus message!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void g3g_dbusmessage::append(dbus_int16_t toAppend)
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return;
	}
    if (!dbus_message_iter_append_basic(mMessageIterList.back(), DBUS_TYPE_INT16, &toAppend))
    {
        std::cout<<"DBusMessageHandler::append Cannot allocate DBus message!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}
void g3g_dbusmessage::append(dbus_int32_t toAppend)
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return;
	}
    if (!dbus_message_iter_append_basic(mMessageIterList.back(), DBUS_TYPE_INT32, &toAppend))
    {
        std::cout<<"DBusMessageHandler::append Cannot allocate DBus message!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}
void g3g_dbusmessage::append(dbus_int64_t toAppend)
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return;
	}
    if (!dbus_message_iter_append_basic(mMessageIterList.back(), DBUS_TYPE_INT64, &toAppend))
    {
        std::cout<<"DBusMessageHandler::append Cannot allocate DBus message!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void g3g_dbusmessage::append(dbus_uint16_t toAppend)
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return;
	}
    if (!dbus_message_iter_append_basic(mMessageIterList.back(), DBUS_TYPE_UINT16, &toAppend))
    {
        std::cout<<"DBusMessageHandler::append Cannot allocate DBus message!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}
void g3g_dbusmessage::append(dbus_uint32_t toAppend)
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return;
	}
    if (!dbus_message_iter_append_basic(mMessageIterList.back(), DBUS_TYPE_UINT32, &toAppend))
    {
        std::cout<<"DBusMessageHandler::append Cannot allocate DBus message!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}
void g3g_dbusmessage::append(dbus_uint64_t toAppend)
{
	if(mMessageIterList.empty())
	{
		std::cout<<"InCorrect Use, Message Iterator not initialized"<<std::endl;
		return;
	}
    if (!dbus_message_iter_append_basic(mMessageIterList.back(), DBUS_TYPE_UINT64, &toAppend))
    {
        std::cout<<"DBusMessageHandler::append Cannot allocate DBus message!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void g3g_dbusmessage::append(const char* toAppend)
{
	if(mMessageIterList.empty())
	{

	}
    if (!dbus_message_iter_append_basic(mMessageIterList.back(), DBUS_TYPE_STRING, &toAppend))
    {
        std::cout<<"DBusMessageHandler::append Cannot allocate DBus message!"<<std::endl;
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void g3g_dbusmessage::append(std::string& toAppend)
{
	append(toAppend.c_str());
}

bool g3g_dbusmessage::bRecurseSubContainer()
{
	switch(dbus_message_iter_get_arg_type(mMessageIterList.back()))
	{
	case CONTAINER_TYPE_ARRAY:
	case CONTAINER_TYPE_VARIANT:
	case CONTAINER_TYPE_STRUCT:
	case CONTAINER_TYPE_DICT_ENTRY:
	{
		DBusMessageIter *subiter = new DBusMessageIter;
		dbus_message_iter_recurse(mMessageIterList.back(),subiter);
		mMessageIterList.push_back(subiter);
		//subiter = NULL;//Satisfying lint
		return true;
	}
	break;
	default:
		std::cout<<"Item is not a sub container, type is "<<dbus_message_iter_get_arg_type(mMessageIterList.back());
		break;
	}
	return false;
}
bool g3g_dbusmessage::bCloseSubContainer_ReadFinished()
{
	DBusMessageIter *subiter = mMessageIterList.back();
	if(NULL != subiter)
	{
		delete subiter;
	}
	mMessageIterList.pop_back();

	dbus_message_iter_next(mMessageIterList.back());//Go to the next element

	return true;
}

bool g3g_dbusmessage::bSubContainerEnd()
{
	return (DBUS_TYPE_INVALID == dbus_message_iter_get_arg_type(mMessageIterList.back()));
}
bool g3g_dbusmessage::bCreateSubContainer(T_e8_ContainerType eType,const char* signature)
{
	dbus_bool_t success = false;

	if(!mMessageIterList.empty())
	{
		switch(eType)
		{
		case CONTAINER_TYPE_ARRAY:
		case CONTAINER_TYPE_VARIANT:
		case CONTAINER_TYPE_STRUCT:
		case CONTAINER_TYPE_DICT_ENTRY:
		{
			DBusMessageIter *subiter = new DBusMessageIter;
			success = dbus_message_iter_open_container(mMessageIterList.back(), eType, signature, subiter);
			if(success)
			{
				mMessageIterList.push_back(subiter);
			}
			else
			{
				delete subiter;
			}
			//subiter = NULL;//Satisfying lint
		}	break;
		default:
			std::cout<<"Container Type is Invalid"<<std::endl;
			break;
		}
	}
	else
	{
		std::cout<<"Message Iter list is empty"<<std::endl;
	}
	return (bool)success;
}
bool g3g_dbusmessage::bCloseSubContainer_WriteFinished()
{
	dbus_bool_t success = false;

	if(mMessageIterList.size() > 1 )
	{
		DBusMessageIter* subiter = mMessageIterList.back();
		mMessageIterList.pop_back();
		success = dbus_message_iter_close_container(mMessageIterList.back(),subiter);
		delete subiter;
	}
	else
	{
		std::cout<<"No Subcontainer left to close"<<std::endl;
	}
	return success;
}

g3g_dbusmessage::~g3g_dbusmessage()
{
	vClearMessageIterList();
}
